<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Models\AuditLog;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Validation\ValidationException;

class LoginController extends Controller
{
    /**
     * Show login form.
     */
    public function showLoginForm()
    {
        return view('auth.login');
    }

    /**
     * Handle login request.
     */
    public function login(Request $request)
    {
        $credentials = $request->validate([
            'email' => ['required', 'email'],
            'password' => ['required'],
        ]);

        // Check if user exists
        $user = \App\Models\User::where('email', $credentials['email'])->first();

        if (!$user) {
            // Log failed login - unknown user
            AuditLog::log(
                'Login Failed',
                'users',
                null,
                null,
                ['email' => $credentials['email'], 'reason' => 'User not found']
            );

            throw ValidationException::withMessages([
                'email' => __('The provided credentials do not match our records.'),
            ]);
        }

        if (!$user->is_active) {
            // Log failed login - inactive user
            AuditLog::log(
                'Login Failed',
                'users',
                $user->id,
                null,
                ['email' => $credentials['email'], 'reason' => 'Account deactivated']
            );

            throw ValidationException::withMessages([
                'email' => __('Your account has been deactivated.'),
            ]);
        }

        if (Auth::attempt($credentials, $request->boolean('remember'))) {
            $request->session()->regenerate();

            // Log successful login
            AuditLog::log(
                'Login',
                'users',
                Auth::id(),
                null,
                ['email' => $credentials['email']]
            );

            return redirect()->intended(route('dashboard'));
        }

        // Log failed login - wrong password
        AuditLog::log(
            'Login Failed',
            'users',
            $user->id,
            null,
            ['email' => $credentials['email'], 'reason' => 'Invalid password']
        );

        throw ValidationException::withMessages([
            'email' => __('The provided credentials do not match our records.'),
        ]);
    }

    /**
     * Handle logout request.
     */
    public function logout(Request $request)
    {
        $userId = Auth::id();
        $userEmail = Auth::user()?->email;

        Auth::logout();

        // Log logout
        AuditLog::log(
            'Logout',
            'users',
            $userId,
            null,
            ['email' => $userEmail]
        );

        $request->session()->invalidate();
        $request->session()->regenerateToken();

        return redirect('/');
    }
}

