<?php

namespace App\Http\Controllers;

use App\Models\Account;
use App\Models\Contribution;
use App\Models\Member;
use App\Models\Organization;
use App\Models\Pledge;
use App\Models\Transaction;
use App\Models\TransactionCategory;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class ContributionController extends Controller
{
    public function __construct()
    {
        $this->middleware('permission:contributions.view')->only(['index', 'show']);
        $this->middleware('permission:contributions.create')->only(['create', 'store']);
        $this->middleware('permission:contributions.delete')->only(['destroy']);
    }

    protected function getOrganizationScope()
    {
        $user = Auth::user();
        if ($user->hasRole('Super Admin')) {
            return null;
        }
        $orgIds = collect([$user->organization_id]);
        $this->getChildOrganizations($user->organization_id, $orgIds);
        return $orgIds->toArray();
    }

    protected function getChildOrganizations($parentId, &$orgIds)
    {
        $children = Organization::where('parent_id', $parentId)->pluck('id');
        foreach ($children as $childId) {
            $orgIds->push($childId);
            $this->getChildOrganizations($childId, $orgIds);
        }
    }

    public function index(Request $request)
    {
        $query = Contribution::with(['member', 'transaction.account', 'pledge']);
        $scope = $this->getOrganizationScope();
        
        if ($scope !== null) {
            $query->whereHas('member', fn($q) => $q->whereIn('organization_id', $scope));
        }

        if ($request->filled('search')) {
            $query->whereHas('member', fn($q) => 
                $q->where('first_name', 'like', "%{$request->search}%")
                  ->orWhere('last_name', 'like', "%{$request->search}%")
            );
        }

        if ($request->filled('date_from')) {
            $query->whereDate('date', '>=', $request->date_from);
        }

        if ($request->filled('date_to')) {
            $query->whereDate('date', '<=', $request->date_to);
        }

        $contributions = $query->orderByDesc('date')->paginate(20)->withQueryString();

        return view('payments.contributions.index', compact('contributions'));
    }

    public function create(Request $request)
    {
        $scope = $this->getOrganizationScope();
        
        $members = Member::query();
        if ($scope !== null) {
            $members->whereIn('organization_id', $scope);
        }
        $members = $members->orderBy('first_name')->get();

        $accounts = Account::query();
        if ($scope !== null) {
            $accounts->whereIn('organization_id', $scope);
        }
        $accounts = $accounts->orderBy('name')->get();

        $categories = TransactionCategory::where('type', 'Income')->orderBy('name')->get();

        // Get member's pending pledges if member selected
        $pledges = collect();
        if ($request->filled('member_id')) {
            $pledges = Pledge::where('member_id', $request->member_id)
                ->whereIn('status', ['Pending', 'Partial'])
                ->get();
        }

        return view('payments.contributions.create', compact('members', 'accounts', 'categories', 'pledges'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'member_id' => 'required|exists:members,id',
            'account_id' => 'required|exists:accounts,id',
            'category_id' => 'required|exists:transaction_categories,id',
            'amount' => 'required|numeric|min:1',
            'purpose' => 'required|string|max:255',
            'date' => 'required|date',
            'pledge_id' => 'nullable|exists:pledges,id',
            'reference' => 'nullable|string|max:100',
        ]);

        // Check scope
        $member = Member::findOrFail($validated['member_id']);
        $account = Account::findOrFail($validated['account_id']);
        $scope = $this->getOrganizationScope();
        
        if ($scope !== null) {
            if (!in_array($member->organization_id, $scope)) abort(403);
            if (!in_array($account->organization_id, $scope)) abort(403);
        }

        DB::transaction(function () use ($validated, $account, $member) {
            // Create income transaction (auto-approved since it's Credit)
            $transaction = Transaction::create([
                'organization_id' => $account->organization_id,
                'account_id' => $validated['account_id'],
                'category_id' => $validated['category_id'],
                'amount' => $validated['amount'],
                'type' => 'Credit',
                'status' => 'Approved',
                'reference' => $validated['reference'] ?? 'CONTRIB-' . time(),
                'description' => "Contribution from {$member->full_name}: {$validated['purpose']}",
                'date' => $validated['date'],
                'recorded_by' => Auth::id(),
            ]);

            // Create contribution record
            Contribution::create([
                'member_id' => $validated['member_id'],
                'transaction_id' => $transaction->id,
                'pledge_id' => $validated['pledge_id'] ?? null,
                'amount' => $validated['amount'],
                'purpose' => $validated['purpose'],
                'date' => $validated['date'],
            ]);
        });

        return redirect()->route('contributions.index')->with('success', 'Contribution recorded successfully.');
    }

    public function show(Contribution $contribution)
    {
        $scope = $this->getOrganizationScope();
        if ($scope !== null && !in_array($contribution->member->organization_id, $scope)) {
            abort(403);
        }

        $contribution->load(['member.organization', 'transaction.account', 'pledge']);

        return view('payments.contributions.show', compact('contribution'));
    }
}
