<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;

class DashboardController extends Controller
{
    /**
     * Show the main dashboard.
     */
    public function index()
    {
        $user = auth()->user();

        // 1. Members Stats
        $totalMembers = \App\Models\Member::count();
        $recentMembers = \App\Models\Member::latest()->take(5)->get();

        // 2. Financial Stats (Current Fiscal Year)
        // Global totals (simple sum of all transactions)
        $totalIncome = \App\Models\Transaction::where('type', 'Credit')->where('status', 'Approved')->sum('amount');
        $totalExpense = \App\Models\Transaction::where('type', 'Debit')->where('status', 'Approved')->sum('amount');
        
        $orgFinancials = null;
        if ($user->hasRole('Super Admin')) {
            // Fetch all orgs with their DIRECT income/expense/members
            $allOrgs = \App\Models\Organization::withSum(['transactions as direct_income' => function($q) {
                $q->where('type', 'Credit')->where('status', 'Approved');
            }], 'amount')
            ->withSum(['transactions as direct_expense' => function($q) {
                $q->where('type', 'Debit')->where('status', 'Approved');
            }], 'amount')
            ->withCount('members')
            ->get();

            // Build Tree and Aggregate
            $orgFinancials = $this->buildOrgTree($allOrgs);
        }
        
        // 3. Pending Approvals
        $pendingApprovals = \App\Models\Transaction::where('status', 'Pending')->count();

        // 4. Audit Logs (Recent Activity)
        $recentActivity = \App\Models\AuditLog::with('user')->latest()->take(6)->get();

        return view('dashboard.index', compact(
            'user', 
            'totalMembers', 
            'recentMembers', 
            'totalIncome', 
            'totalExpense', 
            'pendingApprovals',
            'recentActivity',
            'orgFinancials'
        ));
    }

    /**
     * Build organization tree with accumulated totals.
     */
    private function buildOrgTree($allOrgs)
    {
        // 1. Convert all models to array nodes first
        $nodesById = [];
        foreach ($allOrgs as $org) {
            $nodesById[$org->id] = [
                'id' => $org->id,
                'name' => $org->name,
                'parent_id' => $org->parent_id,
                'total_income' => (float) ($org->direct_income ?? 0),
                'total_expense' => (float) ($org->direct_expense ?? 0),
                'total_members' => (int) ($org->members_count ?? 0),
                'children_list' => []
            ];
        }

        $tree = [];

        // 2. Build Hierarchy using references
        foreach ($nodesById as $id => &$node) {
            if ($node['parent_id'] && isset($nodesById[$node['parent_id']])) {
                $nodesById[$node['parent_id']]['children_list'][] = &$node;
            } else {
                $tree[] = &$node;
            }
        }
        unset($node); // Break reference

        // 3. Aggregate Totals (Bottom-up)
        foreach ($tree as &$root) {
            $this->aggregateTotals($root);
        }

        return $tree;
    }

    private function aggregateTotals(&$node)
    {
        foreach ($node['children_list'] as &$child) {
            $this->aggregateTotals($child);
            $node['total_income'] += $child['total_income'];
            $node['total_expense'] += $child['total_expense'];
            $node['total_members'] += $child['total_members'];
        }
    }
}
