<?php

namespace App\Http\Controllers;

use App\Models\Member;
use App\Models\Organization;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class MemberController extends Controller
{
    public function __construct()
    {
        $this->middleware('permission:members.view')->only(['index', 'show']);
        $this->middleware('permission:members.create')->only(['create', 'store']);
        $this->middleware('permission:members.edit')->only(['edit', 'update']);
        $this->middleware('permission:members.delete')->only(['destroy']);
        $this->middleware('permission:members.export')->only(['export']);
    }

    /**
     * Get organization IDs within the user's scope.
     */
    protected function getOrganizationScope()
    {
        $user = Auth::user();
        
        // Super Admin sees all
        if ($user->hasRole('Super Admin')) {
            return null; // No filter
        }
        
        // Get user's organization and all children recursively
        $orgIds = collect([$user->organization_id]);
        $this->getChildOrganizations($user->organization_id, $orgIds);
        
        return $orgIds->toArray();
    }
    
    /**
     * Recursively get child organization IDs.
     */
    protected function getChildOrganizations($parentId, &$orgIds)
    {
        $children = Organization::where('parent_id', $parentId)->pluck('id');
        foreach ($children as $childId) {
            $orgIds->push($childId);
            $this->getChildOrganizations($childId, $orgIds);
        }
    }

    /**
     * Display a listing of members.
     */
    public function index(Request $request)
    {
        $query = Member::with(['organization', 'user']);
        
        // Apply scope filtering
        $scope = $this->getOrganizationScope();
        if ($scope !== null) {
            $query->whereIn('organization_id', $scope);
        }
        
        // Search
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('first_name', 'like', "%{$search}%")
                  ->orWhere('last_name', 'like', "%{$search}%")
                  ->orWhere('email', 'like', "%{$search}%")
                  ->orWhere('phone', 'like', "%{$search}%");
            });
        }
        
        // Filter by status
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }
        
        // Filter by category
        if ($request->filled('category')) {
            $query->where('category', $request->category);
        }
        
        // Filter by organization
        if ($request->filled('organization_id')) {
            $query->where('organization_id', $request->organization_id);
        }
        
        $members = $query->orderBy('first_name')->paginate(15)->withQueryString();
        
        // Get organizations for filter dropdown (within scope)
        $organizations = Organization::query();
        if ($scope !== null) {
            $organizations->whereIn('id', $scope);
        }
        $organizations = $organizations->orderBy('name')->get();
        
        return view('members.index', compact('members', 'organizations'));
    }

    /**
     * Show the form for creating a new member.
     */
    public function create()
    {
        $scope = $this->getOrganizationScope();
        
        $organizations = Organization::query();
        if ($scope !== null) {
            $organizations->whereIn('id', $scope);
        }
        $organizations = $organizations->orderBy('name')->get();
        
        return view('members.create', compact('organizations'));
    }

    /**
     * Store a newly created member.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'organization_id' => 'required|exists:organizations,id',
            'first_name' => 'required|string|max:255',
            'last_name' => 'required|string|max:255',
            'phone' => 'nullable|string|max:20',
            'email' => 'nullable|email|max:255',
            'dob' => 'nullable|date',
            'gender' => 'nullable|in:Male,Female',
            'marital_status' => 'nullable|string|max:50',
            'profession' => 'nullable|string|max:100',
            'category' => 'required|in:Professional,Student,Entrepreneur,Other',
            'status' => 'required|in:Active,Inactive,Suspended,Left',
        ]);
        
        // Verify user has access to this organization
        $scope = $this->getOrganizationScope();
        if ($scope !== null && !in_array($validated['organization_id'], $scope)) {
            abort(403, 'You do not have access to this organization.');
        }
        
        Member::create($validated);
        
        return redirect()->route('members.index')
            ->with('success', 'Member registered successfully.');
    }

    /**
     * Display the specified member.
     */
    public function show(Member $member)
    {
        // Verify access
        $scope = $this->getOrganizationScope();
        if ($scope !== null && !in_array($member->organization_id, $scope)) {
            abort(403);
        }
        
        $member->load(['organization', 'user']);
        
        return view('members.show', compact('member'));
    }

    /**
     * Show the form for editing the specified member.
     */
    public function edit(Member $member)
    {
        $scope = $this->getOrganizationScope();
        if ($scope !== null && !in_array($member->organization_id, $scope)) {
            abort(403);
        }
        
        $organizations = Organization::query();
        if ($scope !== null) {
            $organizations->whereIn('id', $scope);
        }
        $organizations = $organizations->orderBy('name')->get();
        
        return view('members.edit', compact('member', 'organizations'));
    }

    /**
     * Update the specified member.
     */
    public function update(Request $request, Member $member)
    {
        $scope = $this->getOrganizationScope();
        if ($scope !== null && !in_array($member->organization_id, $scope)) {
            abort(403);
        }
        
        $validated = $request->validate([
            'organization_id' => 'required|exists:organizations,id',
            'first_name' => 'required|string|max:255',
            'last_name' => 'required|string|max:255',
            'phone' => 'nullable|string|max:20',
            'email' => 'nullable|email|max:255',
            'dob' => 'nullable|date',
            'gender' => 'nullable|in:Male,Female',
            'marital_status' => 'nullable|string|max:50',
            'profession' => 'nullable|string|max:100',
            'category' => 'required|in:Professional,Student,Entrepreneur,Other',
            'status' => 'required|in:Active,Inactive,Suspended,Left',
        ]);
        
        // Verify new organization is in scope
        if ($scope !== null && !in_array($validated['organization_id'], $scope)) {
            abort(403, 'You do not have access to this organization.');
        }
        
        $member->update($validated);
        
        return redirect()->route('members.index')
            ->with('success', 'Member updated successfully.');
    }

    /**
     * Remove the specified member.
     */
    public function destroy(Member $member)
    {
        $scope = $this->getOrganizationScope();
        if ($scope !== null && !in_array($member->organization_id, $scope)) {
            abort(403);
        }
        
        $member->delete();
        
        return redirect()->route('members.index')
            ->with('success', 'Member deleted successfully.');
    }

    /**
     * Export members to CSV.
     */
    public function export(Request $request)
    {
        $query = Member::with(['organization']);
        
        $scope = $this->getOrganizationScope();
        if ($scope !== null) {
            $query->whereIn('organization_id', $scope);
        }
        
        // Apply same filters as index
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }
        if ($request->filled('category')) {
            $query->where('category', $request->category);
        }
        
        $members = $query->orderBy('first_name')->get();
        
        $filename = 'members_' . date('Y-m-d') . '.csv';
        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => "attachment; filename=\"{$filename}\"",
        ];
        
        $callback = function() use ($members) {
            $file = fopen('php://output', 'w');
            
            // Header row
            fputcsv($file, ['First Name', 'Last Name', 'Phone', 'Email', 'Organization', 'Category', 'Status']);
            
            foreach ($members as $member) {
                fputcsv($file, [
                    $member->first_name,
                    $member->last_name,
                    $member->phone,
                    $member->email,
                    $member->organization?->name,
                    $member->category,
                    $member->status,
                ]);
            }
            
            fclose($file);
        };
        
        return response()->stream($callback, 200, $headers);
    }
}
