<?php

namespace App\Http\Controllers;

use App\Models\Level;
use App\Models\Member;
use App\Models\Organization;
use App\Models\OrganizationLeader;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class OrganizationController extends Controller
{
    public function __construct()
    {
        $this->middleware('permission:organizations.view')->only(['index', 'show', 'leaders']);
        $this->middleware('permission:organizations.create')->only(['create', 'store', 'createChild']);
        $this->middleware('permission:organizations.edit')->only(['edit', 'update']);
        $this->middleware('permission:organizations.delete')->only(['destroy']);
        $this->middleware('permission:organizations.leaders')->only(['leaders', 'addLeader', 'removeLeader']);
    }

    protected function getOrganizationScope()
    {
        $user = Auth::user();
        if ($user->hasRole('Super Admin')) {
            return null;
        }
        $orgIds = collect([$user->organization_id]);
        if ($user->organization) {
            $orgIds = $user->organization->getDescendantIds();
            $orgIds->push($user->organization_id);
        }
        return $orgIds->toArray();
    }

    public function index(Request $request)
    {
        // Get root organizations (national level or no parent)
        $query = Organization::with(['level', 'children.level'])
            ->whereNull('parent_id')
            ->orWhere(function($q) {
                $q->whereHas('level', fn($l) => $l->where('slug', 'national'));
            });

        $scope = $this->getOrganizationScope();
        if ($scope !== null) {
            $query->whereIn('id', $scope);
        }

        $organizations = $query->orderBy('name')->get();
        $levels = Level::orderBy('id')->get();

        return view('organizations.index', compact('organizations', 'levels'));
    }

    public function create(Request $request)
    {
        $scope = $this->getOrganizationScope();
        
        $parentOrgs = Organization::with('level');
        if ($scope !== null) {
            $parentOrgs->whereIn('id', $scope);
        }
        $parentOrgs = $parentOrgs->orderBy('name')->get();

        $levels = Level::orderBy('id')->get();

        return view('organizations.create', compact('parentOrgs', 'levels'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'level_id' => 'required|exists:levels,id',
            'parent_id' => 'nullable|exists:organizations,id',
            'code' => 'nullable|string|max:50|unique:organizations,code',
            'location' => 'nullable|string|max:255',
        ]);

        // Check scope for parent
        if ($validated['parent_id']) {
            $scope = $this->getOrganizationScope();
            if ($scope !== null && !in_array($validated['parent_id'], $scope)) {
                abort(403);
            }
        }

        Organization::create($validated);

        return redirect()->route('organizations.index')->with('success', 'Organization created successfully.');
    }

    public function show(Organization $organization)
    {
        $scope = $this->getOrganizationScope();
        if ($scope !== null && !in_array($organization->id, $scope)) {
            abort(403);
        }

        $organization->load([
            'level', 
            'parent.level', 
            'children.level',
            'currentLeaders.member',
            'members'
        ]);

        $members = Member::all(); // For adding leaders
        $positions = OrganizationLeader::positions();

        return view('organizations.show', compact('organization', 'members', 'positions'));
    }

    public function edit(Organization $organization)
    {
        $scope = $this->getOrganizationScope();
        if ($scope !== null && !in_array($organization->id, $scope)) {
            abort(403);
        }

        $parentOrgs = Organization::where('id', '!=', $organization->id)
            ->with('level')
            ->orderBy('name')
            ->get();
        $levels = Level::orderBy('id')->get();

        return view('organizations.edit', compact('organization', 'parentOrgs', 'levels'));
    }

    public function update(Request $request, Organization $organization)
    {
        $scope = $this->getOrganizationScope();
        if ($scope !== null && !in_array($organization->id, $scope)) {
            abort(403);
        }

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'level_id' => 'required|exists:levels,id',
            'parent_id' => 'nullable|exists:organizations,id',
            'code' => 'nullable|string|max:50|unique:organizations,code,' . $organization->id,
            'location' => 'nullable|string|max:255',
        ]);

        // Prevent circular reference
        if ($validated['parent_id'] == $organization->id) {
            return back()->with('error', 'Organization cannot be its own parent.');
        }

        $organization->update($validated);

        return redirect()->route('organizations.show', $organization)->with('success', 'Organization updated.');
    }

    public function destroy(Organization $organization)
    {
        $scope = $this->getOrganizationScope();
        if ($scope !== null && !in_array($organization->id, $scope)) {
            abort(403);
        }

        if ($organization->children()->exists()) {
            return back()->with('error', 'Cannot delete organization with sub-organizations.');
        }

        if ($organization->members()->exists()) {
            return back()->with('error', 'Cannot delete organization with members. Migrate members first.');
        }

        $organization->delete();

        return redirect()->route('organizations.index')->with('success', 'Organization deleted.');
    }

    /**
     * Add a leader to organization.
     */
    public function addLeader(Request $request, Organization $organization)
    {
        $scope = $this->getOrganizationScope();
        if ($scope !== null && !in_array($organization->id, $scope)) {
            abort(403);
        }

        $validated = $request->validate([
            'member_id' => 'required|exists:members,id',
            'position' => 'required|string|max:100',
            'started_at' => 'required|date',
        ]);

        $validated['organization_id'] = $organization->id;

        OrganizationLeader::create($validated);

        return back()->with('success', 'Leader added to board.');
    }

    /**
     * Remove a leader from organization.
     */
    public function removeLeader(Organization $organization, OrganizationLeader $leader)
    {
        $scope = $this->getOrganizationScope();
        if ($scope !== null && !in_array($organization->id, $scope)) {
            abort(403);
        }

        // Instead of deleting, set end date
        $leader->update(['ended_at' => now()]);

        return back()->with('success', 'Leader removed from board.');
    }

    /**
     * Get children of an organization (for AJAX).
     */
    public function children(Organization $organization)
    {
        return response()->json(
            $organization->children()->with('level')->get()
        );
    }
}
