<?php

namespace App\Http\Controllers;

use App\Models\Member;
use App\Models\Organization;
use App\Models\Pledge;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class PledgeController extends Controller
{
    public function __construct()
    {
        $this->middleware('permission:pledges.view')->only(['index', 'show']);
        $this->middleware('permission:pledges.create')->only(['create', 'store']);
        $this->middleware('permission:pledges.edit')->only(['edit', 'update']);
        $this->middleware('permission:pledges.delete')->only(['destroy']);
    }

    protected function getOrganizationScope()
    {
        $user = Auth::user();
        if ($user->hasRole('Super Admin')) {
            return null;
        }
        $orgIds = collect([$user->organization_id]);
        $this->getChildOrganizations($user->organization_id, $orgIds);
        return $orgIds->toArray();
    }

    protected function getChildOrganizations($parentId, &$orgIds)
    {
        $children = Organization::where('parent_id', $parentId)->pluck('id');
        foreach ($children as $childId) {
            $orgIds->push($childId);
            $this->getChildOrganizations($childId, $orgIds);
        }
    }

    public function index(Request $request)
    {
        $query = Pledge::with('member.organization');
        $scope = $this->getOrganizationScope();
        
        if ($scope !== null) {
            $query->whereHas('member', fn($q) => $q->whereIn('organization_id', $scope));
        }

        if ($request->filled('search')) {
            $query->whereHas('member', fn($q) => 
                $q->where('first_name', 'like', "%{$request->search}%")
                  ->orWhere('last_name', 'like', "%{$request->search}%")
            );
        }

        if ($request->filled('year')) {
            $query->where('year', $request->year);
        }

        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        $pledges = $query->orderByDesc('year')->orderByDesc('id')->paginate(20)->withQueryString();

        return view('payments.pledges.index', compact('pledges'));
    }

    public function create()
    {
        $scope = $this->getOrganizationScope();
        $members = Member::query();
        if ($scope !== null) {
            $members->whereIn('organization_id', $scope);
        }
        $members = $members->orderBy('first_name')->get();

        return view('payments.pledges.create', compact('members'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'member_id' => 'required|exists:members,id',
            'year' => 'required|integer|min:2020|max:2100',
            'purpose' => 'required|string|max:255',
            'amount' => 'required|numeric|min:1',
        ]);

        // Check scope
        $member = Member::findOrFail($validated['member_id']);
        $scope = $this->getOrganizationScope();
        if ($scope !== null && !in_array($member->organization_id, $scope)) {
            abort(403);
        }

        $validated['status'] = 'Pending';
        $validated['paid_amount'] = 0;

        Pledge::create($validated);

        return redirect()->route('pledges.index')->with('success', 'Pledge created successfully.');
    }

    public function show(Pledge $pledge)
    {
        $scope = $this->getOrganizationScope();
        if ($scope !== null && !in_array($pledge->member->organization_id, $scope)) {
            abort(403);
        }

        $pledge->load(['member.organization', 'contributions.transaction']);

        return view('payments.pledges.show', compact('pledge'));
    }

    public function edit(Pledge $pledge)
    {
        $scope = $this->getOrganizationScope();
        if ($scope !== null && !in_array($pledge->member->organization_id, $scope)) {
            abort(403);
        }

        $members = Member::query();
        if ($scope !== null) {
            $members->whereIn('organization_id', $scope);
        }
        $members = $members->orderBy('first_name')->get();

        return view('payments.pledges.edit', compact('pledge', 'members'));
    }

    public function update(Request $request, Pledge $pledge)
    {
        $scope = $this->getOrganizationScope();
        if ($scope !== null && !in_array($pledge->member->organization_id, $scope)) {
            abort(403);
        }

        $validated = $request->validate([
            'year' => 'required|integer|min:2020|max:2100',
            'purpose' => 'required|string|max:255',
            'amount' => 'required|numeric|min:1',
        ]);

        $pledge->update($validated);
        $pledge->updateStatus();

        return redirect()->route('pledges.index')->with('success', 'Pledge updated successfully.');
    }

    public function destroy(Pledge $pledge)
    {
        $scope = $this->getOrganizationScope();
        if ($scope !== null && !in_array($pledge->member->organization_id, $scope)) {
            abort(403);
        }

        if ($pledge->contributions()->exists()) {
            return back()->with('error', 'Cannot delete pledge with existing contributions.');
        }

        $pledge->delete();

        return redirect()->route('pledges.index')->with('success', 'Pledge deleted successfully.');
    }
}
