<?php

namespace App\Models;

use App\Traits\Auditable;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\SoftDeletes;

class Budget extends Model
{
    use HasFactory, Auditable, SoftDeletes;

    protected $fillable = [
        'organization_id',
        'fiscal_year',
        'total_income',
        'total_expense',
        'status',
    ];

    protected function casts(): array
    {
        return [
            'total_income' => 'decimal:2',
            'total_expense' => 'decimal:2',
        ];
    }

    /**
     * Get the net balance (income - expense).
     */
    public function getNetBalanceAttribute(): float
    {
        return $this->total_income - $this->total_expense;
    }

    /**
     * Recalculate totals from budget items.
     */
    public function recalculateTotals(): void
    {
        $this->total_income = $this->items()
            ->whereHas('category', fn($q) => $q->where('type', 'Income'))
            ->sum('amount');

        $this->total_expense = $this->items()
            ->whereHas('category', fn($q) => $q->where('type', 'Expense'))
            ->sum('amount');

        $this->save();
    }

    public function organization(): BelongsTo
    {
        return $this->belongsTo(Organization::class);
    }

    public function items(): HasMany
    {
        return $this->hasMany(BudgetItem::class);
    }

    public function creator(): BelongsTo
    {
        return $this->belongsTo(User::class, 'creator_id');
    }

    public function approver(): BelongsTo
    {
        return $this->belongsTo(User::class, 'approver_id');
    }
}
