<?php

namespace Database\Seeders;

use App\Models\Level;
use App\Models\Organization;
use Illuminate\Database\Seeder;

class OrganizationSeeder extends Seeder
{
    /**
     * Run the database seeds.
     * 
     * Creates hierarchy: ATAPE Tanzania → 2 Unions → Conferences → Zones → Local Churches
     */
    public function run(): void
    {
        // Get level IDs
        $nationalId = Level::where('slug', 'national')->first()?->id ?? 1;
        $unionId = Level::where('slug', 'union')->first()?->id ?? 2;
        $conferenceId = Level::where('slug', 'conference')->first()?->id ?? 3;
        $zoneId = Level::where('slug', 'zone')->first()?->id ?? 4;
        $localId = Level::where('slug', 'local')->first()?->id ?? 5;

        // 1. National Level - ATAPE Tanzania
        $national = Organization::firstOrCreate(
            ['code' => 'ATAPE-TZ'],
            [
                'name' => 'ATAPE Tanzania',
                'level_id' => $nationalId,
                'location' => 'Dar es Salaam',
            ]
        );

        // 2. Union Level - East and South Unions
        $unions = [
            ['name' => 'East Union', 'code' => 'ATAPE-EAST', 'location' => 'Dar es Salaam'],
            ['name' => 'South Union', 'code' => 'ATAPE-SOUTH', 'location' => 'Iringa'],
        ];

        foreach ($unions as $unionData) {
            $union = Organization::firstOrCreate(
                ['code' => $unionData['code']],
                [
                    'name' => $unionData['name'],
                    'level_id' => $unionId,
                    'parent_id' => $national->id,
                    'location' => $unionData['location'],
                ]
            );

            // 3. Conference Level - Multiple per Union
            $conferences = $this->getConferencesForUnion($unionData['name']);
            
            foreach ($conferences as $confData) {
                $conference = Organization::firstOrCreate(
                    ['code' => $confData['code']],
                    [
                        'name' => $confData['name'],
                        'level_id' => $conferenceId,
                        'parent_id' => $union->id,
                        'location' => $confData['location'],
                    ]
                );

                // 4. Zone Level - Multiple per Conference
                $zones = $this->getZonesForConference($confData['name']);
                
                foreach ($zones as $zoneData) {
                    $zone = Organization::firstOrCreate(
                        ['code' => $zoneData['code']],
                        [
                            'name' => $zoneData['name'],
                            'level_id' => $zoneId,
                            'parent_id' => $conference->id,
                            'location' => $zoneData['location'],
                        ]
                    );

                    // 5. Local Level - Churches per Zone
                    $churches = $this->getChurchesForZone($zoneData['name']);
                    
                    foreach ($churches as $churchData) {
                        Organization::firstOrCreate(
                            ['code' => $churchData['code']],
                            [
                                'name' => $churchData['name'],
                                'level_id' => $localId,
                                'parent_id' => $zone->id,
                                'location' => $churchData['location'],
                            ]
                        );
                    }
                }
            }
        }
    }

    protected function getConferencesForUnion(string $unionName): array
    {
        if (str_contains($unionName, 'East')) {
            return [
                ['name' => 'Dar es Salaam Conference', 'code' => 'CONF-DAR', 'location' => 'Dar es Salaam'],
                ['name' => 'Morogoro Conference', 'code' => 'CONF-MOR', 'location' => 'Morogoro'],
                ['name' => 'Tanga Conference', 'code' => 'CONF-TNG', 'location' => 'Tanga'],
                ['name' => 'Pwani Conference', 'code' => 'CONF-PWN', 'location' => 'Pwani'],
                ['name' => 'Zanzibar Conference', 'code' => 'CONF-ZNZ', 'location' => 'Zanzibar'],
                ['name' => 'Dodoma Conference', 'code' => 'CONF-DOD', 'location' => 'Dodoma'],
            ];
        }
        
        return [
            ['name' => 'Iringa Conference', 'code' => 'CONF-IRG', 'location' => 'Iringa'],
            ['name' => 'Mbeya Conference', 'code' => 'CONF-MBY', 'location' => 'Mbeya'],
            ['name' => 'Njombe Conference', 'code' => 'CONF-NJB', 'location' => 'Njombe'],
            ['name' => 'Songea Conference', 'code' => 'CONF-SNG', 'location' => 'Songea'],
            ['name' => 'Rukwa Conference', 'code' => 'CONF-RKW', 'location' => 'Rukwa'],
            ['name' => 'Katavi Conference', 'code' => 'CONF-KTV', 'location' => 'Katavi'],
        ];
    }

    protected function getZonesForConference(string $confName): array
    {
        $prefix = explode(' ', $confName)[0]; // Get city/region name
        
        return [
            ['name' => $prefix . ' Zone', 'code' => 'ZN-' . strtoupper(substr($prefix, 0, 3)), 'location' => $prefix],
            ['name' => $prefix . ' North Zone', 'code' => 'ZN-' . strtoupper(substr($prefix, 0, 3)) . 'N', 'location' => $prefix . ' North'],
        ];
    }

    protected function getChurchesForZone(string $zoneName): array
    {
        $prefix = explode(' ', $zoneName)[0]; // Get area name
        
        return [
            ['name' => $prefix . ' Central Church', 'code' => 'LC-' . strtoupper(substr($prefix, 0, 3)) . 'C', 'location' => $prefix . ' Central'],
            ['name' => $prefix . ' Town Church', 'code' => 'LC-' . strtoupper(substr($prefix, 0, 3)) . 'T', 'location' => $prefix . ' Town'],
        ];
    }
}

