@extends('layouts.app')

@section('title', 'Budget Details')

@section('content')
<div class="page-header">
    <div class="header-left">
        <h1>{{ $budget->organization?->name }} - {{ $budget->fiscal_year }}</h1>
        <div style="display: flex; gap: 1rem; align-items: center;">
            <p class="text-muted" style="margin: 0;">Status: 
                @if($budget->status === 'Draft')
                    <span class="badge badge-warning">Draft</span>
                @elseif($budget->status === 'Pending')
                    <span class="badge badge-info">Pending Approval</span>
                @elseif($budget->status === 'Approved')
                    <span class="badge badge-success">Approved</span>
                @elseif($budget->status === 'Amendment Pending')
                    <span class="badge badge-warning">Amendment Requested</span>
                @endif
            </p>
            @if($budget->rejection_reason && $budget->status === 'Draft')
                <div class="alert alert-error" style="margin: 0; padding: 0.25rem 0.5rem; font-size: 0.85rem;">
                    <strong>Rejection Reason:</strong> {{ $budget->rejection_reason }}
                </div>
            @endif
        </div>
    </div>
    <div class="header-actions">
        {{-- Draft: Creator can Submit --}}
        @if($budget->status === 'Draft')
            <form action="{{ route('budgets.submit', $budget) }}" method="POST" style="display:inline;" data-confirm="Submit this budget for approval? You will not be able to edit it afterward.">
                @csrf
                <button type="submit" class="btn btn-primary">📤 Submit for Approval</button>
            </form>
        @endif

        {{-- Pending: Approver (Not Creator) can Approve/Reject --}}
        @if($budget->status === 'Pending')
            @if(Auth::id() !== $budget->creator_id)
                <form action="{{ route('budgets.approve', $budget) }}" method="POST" style="display:inline;" data-confirm="Approve this budget?">
                    @csrf
                    <button type="submit" class="btn btn-success">✅ Approve</button>
                </form>
                <button type="button" class="btn btn-danger" onclick="promptReject()">❌ Reject</button>
                
                <form id="reject-form" action="{{ route('budgets.reject', $budget) }}" method="POST" style="display:none;">
                    @csrf
                    <input type="hidden" name="reason" id="reject-reason">
                </form>
            @else
                <button class="btn btn-secondary" disabled title="You cannot approve your own submission">⏳ Awaiting Approval</button>
            @endif
        @endif

        {{-- Approved: Anyone can Request Amendment --}}
        @if($budget->status === 'Approved')
            <form action="{{ route('budgets.request-amendment', $budget) }}" method="POST" style="display:inline;" data-confirm="Request to unpublish/amend this budget?">
                @csrf
                <button type="submit" class="btn btn-secondary">🔓 Request Amendment</button>
            </form>
        @endif

        {{-- Amendment Pending: Approver can Grant --}}
        @if($budget->status === 'Amendment Pending')
            @if(Auth::id() !== $budget->creator_id)
                <form action="{{ route('budgets.grant-amendment', $budget) }}" method="POST" style="display:inline;" data-confirm="Grant amendment request? Budget will return to Draft.">
                    @csrf
                    <button type="submit" class="btn btn-warning">🔓 Grant Amendment</button>
                </form>
            @else
                 <button class="btn btn-secondary" disabled>⏳ Amendment Pending</button>
            @endif
        @endif

        <a href="{{ route('budgets.index') }}" class="btn btn-secondary">Back to List</a>
    </div>
</div>

<script>
    function promptReject() {
        let reason = prompt("Please provide a reason for rejection:");
        if (reason) {
            document.getElementById('reject-reason').value = reason;
            document.getElementById('reject-form').submit();
        }
    }
</script>

<!-- Summary Cards -->
<div class="stats-grid" style="margin-bottom: 2rem;">
    <div class="stat-card">
        <div class="stat-value text-success">{{ number_format($budget->total_income, 2) }}</div>
        <div class="stat-label">Total Income</div>
    </div>
    <div class="stat-card">
        <div class="stat-value text-danger">{{ number_format($budget->total_expense, 2) }}</div>
        <div class="stat-label">Total Expense</div>
    </div>
    <div class="stat-card">
        <div class="stat-value {{ $budget->net_balance >= 0 ? 'text-success' : 'text-danger' }}">
            {{ number_format($budget->net_balance, 2) }}
        </div>
        <div class="stat-label">Net Balance</div>
    </div>
</div>

<div class="profile-grid">
    <!-- Income Items -->
    <div class="sidebar-card">
        <h3>📈 Planned Income Targets</h3>
        <p class="text-muted" style="font-size: 0.8rem; margin-bottom: 1rem;">These are targets, not actual transactions.</p>
        @forelse($incomeItems as $item)
        <div class="stat-item" style="display: flex; justify-content: space-between; align-items: center;">
            <div>
                <strong>{{ $item->category?->name }}</strong>
                @if($item->description)<br><small class="text-muted">{{ $item->description }}</small>@endif
            </div>
            <div style="display: flex; align-items: center; gap: 0.5rem;">
                <span class="text-success">{{ number_format($item->amount, 2) }}</span>
                @if($budget->status === 'Draft')
                <form action="{{ route('budgets.items.destroy', [$budget, $item]) }}" method="POST" style="display:inline;" data-confirm="Remove this item?">
                    @csrf
                    @method('DELETE')
                    <button type="submit" class="btn btn-sm btn-link">🗑️</button>
                </form>
                @endif
            </div>
        </div>
        @empty
        <p class="text-muted">No income items.</p>
        @endforelse
    </div>

    <!-- Expense Items -->
    <div class="sidebar-card">
        <h3>📉 Planned Expense Targets</h3>
        <p class="text-muted" style="font-size: 0.8rem; margin-bottom: 1rem;">These are targets, not actual transactions.</p>
        @forelse($expenseItems as $item)
        <div class="stat-item" style="display: flex; justify-content: space-between; align-items: center;">
            <div>
                <strong>{{ $item->category?->name }}</strong>
                @if($item->description)<br><small class="text-muted">{{ $item->description }}</small>@endif
            </div>
            <div style="display: flex; align-items: center; gap: 0.5rem;">
                <span class="text-danger">{{ number_format($item->amount, 2) }}</span>
                @if($budget->status === 'Draft')
                <form action="{{ route('budgets.items.destroy', [$budget, $item]) }}" method="POST" style="display:inline;" data-confirm="Remove this item?">
                    @csrf
                    @method('DELETE')
                    <button type="submit" class="btn btn-sm btn-link">🗑️</button>
                </form>
                @endif
            </div>
        </div>
        @empty
        <p class="text-muted">No expense items.</p>
        @endforelse
    </div>
</div>

@if($budget->status === 'Draft')
<!-- Add Item Form -->
<div class="form-card" style="margin-top: 2rem;">
    <h3>Add Budget Item</h3>
    <form action="{{ route('budgets.items.store', $budget) }}" method="POST">
        @csrf
        <div class="form-row">
            <div class="form-group">
                <label for="category_id">Category *</label>
                <select id="category_id" name="category_id" required>
                    <option value="">Select Category...</option>
                    <optgroup label="Income">
                        @foreach($categories->where('type', 'Income') as $cat)
                        <option value="{{ $cat->id }}">{{ $cat->name }}</option>
                        @endforeach
                    </optgroup>
                    <optgroup label="Expense">
                        @foreach($categories->where('type', 'Expense') as $cat)
                        <option value="{{ $cat->id }}">{{ $cat->name }}</option>
                        @endforeach
                    </optgroup>
                </select>
            </div>
            <div class="form-group">
                <label for="amount">Amount *</label>
                <div class="currency-input">
                    <input type="number" id="amount" name="amount" step="0.01" min="0.01" required placeholder="0.00">
                </div>
            </div>
        </div>
        <div class="form-group">
            <label for="description">Description</label>
            <input type="text" id="description" name="description" placeholder="Optional notes...">
        </div>
        <button type="submit" class="btn btn-primary">Add Item</button>
    </form>
</div>
@endif
@endsection
